/** @file   npccharacter.cpp
 * @brief   Implementation of NpcCharacter
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#include "npccharacter.h"
#include "warglobals.h"
#include "soundsamples.h"
#include "www_map.h"
#include "redrawqueue.h"
#include "animplayer.h"
#include "gameanims.h"
#include "AnimId.h"
using namespace eng2d;
using std::vector;

namespace WeWantWar {

///
/// Static members, constants and datatypes
/// =======================================

const int NpcCharacter::RELOAD_COUNTER_INDEX;


///
/// Constructors, destructor and operators
/// ======================================

/** Constructor
 */
NpcCharacter::NpcCharacter() :
  GameObject(),
  m_animationCombine( 0 ),
  m_weapon( Weapon::W_NOWEAPON ),
  iEnergybar( 0 ),
  iEnergybarUpdate( 0 )
{
  m_soundIndices.push_back( -1 );   // idle sound
  m_soundIndices.push_back( SMP_PLAYER_PAIN );
  m_soundIndices.push_back( -1 );   // attack sound
  m_soundIndices.push_back( -1 );   // moving sound
  m_soundIndices.push_back( SMP_CIVILIAN_DIE );
  m_soundIndices.push_back( -1 );   // waking sound
  
  // The gun nose
  this->addCtrlPoint( Vec2D(6,-22) );
  
  // We create a buffer for the energybar
  iEnergybar = create_bitmap( 48, 3 );
}



/** Destructor
 */
NpcCharacter::~NpcCharacter()
{
  if ( iEnergybar != 0 ) {
    destroy_bitmap( iEnergybar );
    iEnergybar = 0;
  }
}




///
/// Public methods
/// ==============


/** Redraws this character
 */
void NpcCharacter::redraw( RedrawQueue* pQ )
{
  if ( this->hidden() == true ) {
    return;
  }

  iEnergybarUpdate -= 1;
  if ( iEnergybarUpdate < 0 ) {
    this->updateEnergybar();
    iEnergybarUpdate = rand() % 40 + 15;
  }
  
  Vec2D p = this->position();
  int posX = p.intX() - Map::scrollX;
  int posY = p.intY() - Map::scrollY;
  int barX = p.intX() - Map::scrollX;
  
  BITMAP* pS = m_animation.currentFrame().asBitmap();
  posX -= pS->w / 2;
  posY -= pS->h / 2;
  
  pQ->addRotatedSprite( RedrawQueue::PRI_ABOVE_NORMAL, posX,posY,
                        itofix( m_angle ), pS );
                        
  // Draw the energy bar
  if ( iEnergybar != 0 ) {
    barX -= iEnergybar->w / 2;
    pQ->add( RedrawQueue::PRI_FLYING_HIGH, barX, posY,
             RedrawQueue::SPTYPE_SPRITE,
             RedrawQueue::BMODE_TRANS, 50,
             iEnergybar );
  }
}



/** Makes sound
 */
void NpcCharacter::makeSound( GameObject::SoundID id ) const
{
  if ( m_soundIndices.at(id) == -1 ) {
    return;
  }
  Sound::playSample( m_soundIndices.at(id), false );
}



/** Handles the bullet hits
 */
bool NpcCharacter::hitByBullet( Bullet* pB )
{
  if ( pB->iOwner == this ) {
    return false;
  }
  
  if ( pB->iOwner != 0 ) {
    int oid = pB->iOwner->objectType();
    if ( oid == ObjectID::TYPE_PLAYER ) {
      return false;
    } else if ( oid == ObjectID::TYPE_ERIC ||
                oid == ObjectID::TYPE_SIMON ||
                oid == ObjectID::TYPE_YOUKO ) {
      return false;
    }
  }
  ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(),
                                         12, Color(130,60,10) );
  WarGlobals::pPartManager->addSystem( pP );
  
  this->causeDamage( pB );
  return true;
}



/** Kills us
 */
void NpcCharacter::kill()
{
  if ( this->state() == GameObject::STATE_LIVING ) {
    this->state( GameObject::STATE_DYING );
    this->setAnimation( m_animationCombine->FindAnim( GameAnims::EDying ),
                        GameAnims::EDying );
  }
}



/** Sets the weapon for this object
 */
void NpcCharacter::setWeapon( Weapon w )
{
  m_weapon = w;
}




///
/// Getter methods
/// ==============

/** Are we reloading our weapon
 */
bool NpcCharacter::reloading() const
{
  if ( this->getCounter( RELOAD_COUNTER_INDEX ) >= 0 ) {
    return true;
  }
  return false;
}



/** Returns the weapon
 */
Weapon NpcCharacter::getWeapon() const
{
  return m_weapon;
}




///
/// Private or Protected methods
/// ============================

/** Changes the currently running animation.
 */
void NpcCharacter::changeAnimation( int animID )
{
  if ( this->getAnimID() == animID ) {
    return;
  }
  this->setAnimation( m_animationCombine->FindAnim( animID ), animID );
}



/** Handles the shooting
 */
void NpcCharacter::handleShooting()
{
  Vec2D gunV( this->getCtrlPoint( GUN_NOSE_CTRLPOINT ) );
  gunV += this->position();
  Weapon::Specs& rW = Weapon::weaponlist.at( m_weapon.id() );
  
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KRifleShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  
  switch ( m_weapon.id() ) {
    case ( Weapon::W_RIFLE ): {
      Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::ERifle );
      WarGlobals::pBulletManager->spawnBullet( pB );
      this->makeSound( GameObject::SND_ATTACK );
      
      AnimPlayer::spawn( flameAnim, gunV, 0 );
      AnimPlayer::spawn( lightAnim, gunV, 0 );
      this->setCounter( NpcCharacter::RELOAD_COUNTER_INDEX, rW.reload );
      break;
    }
    case ( Weapon::W_SHOTGUN ): {
      for ( int i=0; i < 6; i++ ) {
        Bullet* pB = BulletTable::createBullet( this, m_position, Bullet::EShotgun );
        WarGlobals::pBulletManager->spawnBullet( pB );
      }
      this->makeSound( GameObject::SND_ATTACK );
      AnimPlayer::spawn( flameAnim, gunV, 0 );
      AnimPlayer::spawn( lightAnim, gunV, AnimPlayer::DMODE_ADD,
                         30, Vec2D() );
      this->setCounter( NpcCharacter::RELOAD_COUNTER_INDEX, rW.reload );
      break;
    }
  }
}



/** Updates the contents of the energybar
 */
void NpcCharacter::updateEnergybar()
{
  if ( iEnergybar == 0 ) {
    return;
  }
  
  clear_to_color( iEnergybar, makecol(0,0,0) );
  float ratio = static_cast<float>( iEnergybar->w - 2) / 100.0;
  float endPix = ratio * static_cast<float>( this->health() );
  
  int px = 1 + static_cast<int>( endPix );
  int c = makecol(64,230,80);
  if ( this->health() < 50 ) {
    c = makecol( 180,180,70 );
  }
  if ( this->health() < 20 ) {
    c = makecol( 180,80,64 );
  }
  
  hline( iEnergybar, 1,1,px-1, c );
}


} // end of namespace
